/*******************************************************************************
*
* Copyright (c) 2015-2016 Intel Corporation.  All rights reserved.
*
* This software is available to you under a choice of one of two
* licenses.  You may choose to be licensed under the terms of the GNU
* General Public License (GPL) Version 2, available from the file
* COPYING in the main directory of this source tree, or the
* OpenFabrics.org BSD license below:
*
*   Redistribution and use in source and binary forms, with or
*   without modification, are permitted provided that the following
*   conditions are met:
*
*    - Redistributions of source code must retain the above
*	copyright notice, this list of conditions and the following
*	disclaimer.
*
*    - Redistributions in binary form must reproduce the above
*	copyright notice, this list of conditions and the following
*	disclaimer in the documentation and/or other materials
*	provided with the distribution.
*
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
* EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
* MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
* NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
* BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
* ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
* CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
* SOFTWARE.
*
*******************************************************************************/

#include <linux/fs.h>
#include <linux/debugfs.h>
#include <linux/list.h>

#include "i40iw.h"

#ifdef CONFIG_DEBUG_FS

static const char i40iw_driver_name[] = "i40iw";
static struct dentry *i40iw_dbg_root;
static char cmd_buf[64];

/* QP subcommand */
#define QP_ACTIVE               1
#define QP_INFO                 2
#define QP_Q2                   3
#define QP_HOST_CTX             4
#define QP_CHIP_CTX             5
#define QP_CHIP_RAW             6

#define DUMP_HMC_INFO           1
#define DUMP_HMC_INDEXES        2
#define DUMP_HMC_OBJ_INFO       4
#define DUMP_HMC_SD             8
#define DUMP_HMC_ALL            0xf

#define OFFSET_MASK_4K          0x0000000000000fffl

enum mr_cmd_type {
	MR_CMD_LIST,
	MR_CMD_STAG
};

enum dump_mr_type {
	DUMP_MR_DATA,
	DUMP_MR_LEVEL1,
	DUMP_MR_ROOT,
	DUMP_MR_LEAF
};

static const char *const sd_type[] = { "Invalid", "Paged", "Direct" };

static const char *const obj_info_type[] = {
	"QP", "CQ", "SRQ", "HTE", "ARP", "APBVT", "MR", "XF",
	"XFFL", "Q1", "Q1FL", "TIMER", "FSIMC", "FSIAV", "PBLE"
};

static char *mr_types[] = { "MEM", "QP", "CQ", "MW", "FMR", "FMEM" };
static char *i40iw_dbg_dump_buf;
static size_t i40iw_dbg_dump_data_len;
static size_t i40iw_dbg_dump_buffer_len;
static u32 cmdnew;
static u32 cmddone;
#define BUF_ADDR_SIZE 17	/* address plus space */
#define BUF_DATA_ASCII_SIZE (32 * 3 + 2 + 32 + 1)	/* data plus ASCII */
#define I40IW_DUMP_BUF_SIZE             16384
#define I40IW_DUMP_BUF_HALF_FULL        (I40IW_DUMP_BUF_SIZE / 2)

struct hmc_find {
	dma_addr_t paddr;
	bool fnd;
	dma_addr_t bp_pa;
	bool in_vf;
	u32 vf_id;
	int sd_indx;
	bool paged;
	bool in_pd_itself;
	int pd_indx;
	u64 hmc_addr;
	bool obj_info_fnd;
	enum i40iw_hmc_rsrc_type obj_type;
	int obj_indx;
};

/**
 * dbg_vsnprintf -
 * @fmt:
 **/
static void dbg_vsnprintf(char *fmt, ...)
{
	int cnt;
	va_list argp;

	va_start(argp, fmt);
	cnt = vsnprintf(i40iw_dbg_dump_buf + i40iw_dbg_dump_data_len,
			i40iw_dbg_dump_buffer_len - i40iw_dbg_dump_data_len,
			fmt, argp);
	va_end(argp);

	i40iw_dbg_dump_data_len += cnt;
}

/**
 * dump_help -
 * @hdl:
 **/
static void dump_help(struct i40iw_handler *hdl)
{
	dbg_vsnprintf("Dump commands:\n");
	dbg_vsnprintf(" cqp\n");
	dbg_vsnprintf(" hmc info\n");
	dbg_vsnprintf(" mr list\n");
	dbg_vsnprintf(" mr stag\n");
	dbg_vsnprintf(" qp <n> active           <n> is a qp number or 'all'\n");
	dbg_vsnprintf(" qp <n> info\n");
	dbg_vsnprintf(" qp <n> q2\n");
	dbg_vsnprintf(" qp <n> chip_ctx\n");
	dbg_vsnprintf(" qp <n> chip_raw\n");
	dbg_vsnprintf(" sw-stats\n");
	dbg_vsnprintf(" hw-stats\n");
	cmddone = true;
}

/**
 * i40iw_dbg_save_ucontext -
 * @iwdev:
 * @ucontext:
 **/
void i40iw_dbg_save_ucontext(struct i40iw_device *iwdev,
			     struct i40iw_ucontext *ucontext)
{
	struct i40iw_handler *hdl;
	unsigned long flags;

	hdl = container_of(iwdev, struct i40iw_handler, device);
	INIT_LIST_HEAD(&ucontext->uctx_list);

	spin_lock_irqsave(&hdl->uctx_list_lock, flags);
	list_add_tail(&ucontext->uctx_list, &hdl->ucontext_list);
	spin_unlock_irqrestore(&hdl->uctx_list_lock, flags);
}

/**
 * i40iw_dbg_free_ucontext -
 * @ucontext:
 **/
void i40iw_dbg_free_ucontext(struct i40iw_ucontext *ucontext)
{
	struct i40iw_handler *hdl;
	unsigned long flags;

	hdl = container_of(ucontext->iwdev, struct i40iw_handler, device);

	spin_lock_irqsave(&hdl->uctx_list_lock, flags);
	list_del(&ucontext->uctx_list);
	spin_unlock_irqrestore(&hdl->uctx_list_lock, flags);
}

/**
 * dump_to_buffer -
 * @dataptr:
 * @len:
 * @offset:
 * @rowsize:
 * @groupsize:
 * @ascii:
 * @lendumped:
 * @return:
 **/
static int dump_to_buffer(char *dataptr,
			  size_t len,
			  int offset,
			  int rowsize,
			  int groupsize,
			  bool ascii,
			  size_t *lendumped)
{
	char *bufptr;
	size_t bufleft, linelen;
	int dataleft = len;
	int count, thisrow, i;

	bufptr = i40iw_dbg_dump_buf + i40iw_dbg_dump_data_len;
	bufleft = i40iw_dbg_dump_buffer_len - i40iw_dbg_dump_data_len;

	if (rowsize != 16 && rowsize != 32)
		rowsize = 16;

	*lendumped = 0;
	for (i = 0; i < len; i += rowsize) {
		if (bufleft < (BUF_ADDR_SIZE + BUF_DATA_ASCII_SIZE))
			break;
		count = snprintf(bufptr, bufleft, "  (%04x) ", offset);

		bufleft -= count;
		bufptr += count;
		thisrow = min(dataleft, rowsize);
		hex_dump_to_buffer(dataptr, thisrow, rowsize, groupsize,
				   bufptr, BUF_DATA_ASCII_SIZE, ascii);

		*lendumped += thisrow;
		offset += thisrow;
		dataptr += thisrow;
		dataleft -= thisrow;

		linelen = strlen(bufptr);
		bufptr[linelen++] = '\n';
		bufptr += linelen;
		bufleft -= linelen;
	}

	i40iw_dbg_dump_data_len = bufptr - i40iw_dbg_dump_buf;
	return 0;
}

/**
 * virt_mem -
 * @vmem:
 * @membrname:
 * @prefix:
 **/
static void virt_mem(struct i40iw_virt_mem *vmem, char *membrname, char *prefix)
{
	dbg_vsnprintf("%s%s\n", prefix, membrname);
	dbg_vsnprintf("%s  va: 0x%p\n", prefix, vmem->va);
	dbg_vsnprintf("%s  size: 0x%x\n", prefix, vmem->size);
}

/**
 * dma_mem -
 * @addr:
 * @membrname:
 * @prefix:
 **/
static void dma_mem(struct i40iw_dma_mem *addr, char *membrname, char *prefix)
{
	dbg_vsnprintf("%s%s\n", prefix, membrname);
	dbg_vsnprintf("%s  va: 0x%p\n", prefix, addr->va);
	dbg_vsnprintf("%s  pa: 0x%llx\n", prefix, addr->pa);
	dbg_vsnprintf("%s  size: 0x%x\n", prefix, addr->size);
}

/**
 * find_next_resource -
 * @iwdev:
 * @resource_array:
 * @max_resources:
 * @next:
 * @return:
 **/
static int find_next_resource(struct i40iw_device *iwdev,
			      unsigned long *resource_array,
			      u32 max_resources,
			      u32 next)
{
	u32 resource_num;
	unsigned long flags;

	spin_lock_irqsave(&iwdev->resource_lock, flags);
	resource_num = find_next_bit(resource_array, max_resources, next);
	spin_unlock_irqrestore(&iwdev->resource_lock, flags);
	return (resource_num < max_resources) ? resource_num : -2;
}

/**
 * find_next_qp_num -
 * @iwdev:
 * @qp_id:
 * @return:
 **/
static int find_next_qp_num(struct i40iw_device *iwdev, int qp_id)
{
	do {
		++qp_id;
		if ((qp_id == 0) || (qp_id == 1))
			qp_id = 2;
		qp_id = find_next_resource(iwdev, iwdev->allocated_qps,
					   iwdev->max_qp, qp_id);
		if (qp_id < 0)
			break;
	} while (!iwdev->qp_table[qp_id]);

	return qp_id;
}

/**
 * dump_qp_shared -
 * @qp:
 **/
static void dump_qp_shared(struct i40iw_qp *qp)
{
	struct i40iw_sc_qp *qpsh = &qp->sc_qp;
	struct i40iw_qp_uk *qpuk = &qpsh->qp_uk;

	dbg_vsnprintf("    qp_uk: 0x%p\n", &qpsh->qp_uk);
	dbg_vsnprintf("      sq_base: 0x%p\n", qpuk->sq_base);
	dbg_vsnprintf("      rq_base: 0x%p\n", qpuk->rq_base);
	dbg_vsnprintf("      wqe_alloc_reg: 0x%p\n", qpuk->wqe_alloc_reg);
	dbg_vsnprintf("      sq_wrtrk_array: 0x%p\n", qpuk->sq_wrtrk_array);
	dbg_vsnprintf("      rq_wrid_array: 0x%p\n", qpuk->rq_wrid_array);
	dbg_vsnprintf("      shadow_area: 0x%p\n", qpuk->shadow_area);
	dbg_vsnprintf("      push_db: 0x%p\n", qpuk->push_db);
	dbg_vsnprintf("      push_wqe: 0x%p\n", qpuk->push_wqe);
	dbg_vsnprintf("      sq_ring.head: 0x%x tail: 0x%x size: 0x%x\n",
		      qpuk->sq_ring.head, qpuk->sq_ring.tail,
		      qpuk->sq_ring.size);
	dbg_vsnprintf("      rq_ring.head: 0x%x tail: 0x%x size: 0x%x\n",
		      qpuk->rq_ring.head, qpuk->rq_ring.tail,
		      qpuk->rq_ring.size);
	dbg_vsnprintf("      qp_id: 0x%x\n", qpuk->qp_id);
	dbg_vsnprintf("      sq_size: 0x%x\n", qpuk->sq_size);
	dbg_vsnprintf("      rq_size: 0x%x\n", qpuk->rq_size);

	dbg_vsnprintf("      ops: 0x%p\n", &qpuk->ops);
	dbg_vsnprintf("      use_srq: 0x%x\n", qpuk->use_srq);
	dbg_vsnprintf("      swqe_polarity: 0x%x\n", qpuk->swqe_polarity);
	dbg_vsnprintf("      rwqe_polarity: 0x%x\n", qpuk->rwqe_polarity);
	dbg_vsnprintf("      rq_wqe_size: 0x%x\n", qpuk->rq_wqe_size);
	dbg_vsnprintf("      rq_wqe_size_multiplier: 0x%x\n",
		      qpuk->rq_wqe_size_multiplier);
	dbg_vsnprintf("      max_sq_frag_cnt: 0x%x\n", qpuk->max_sq_frag_cnt);
	dbg_vsnprintf("      max_rq_frag_cnt: 0x%x\n", qpuk->max_rq_frag_cnt);

	dbg_vsnprintf("    sq_pa: 0x%llx\n", qpsh->sq_pa);
	dbg_vsnprintf("    rq_pa: 0x%llx\n", qpsh->rq_pa);
	dbg_vsnprintf("    hw_host_ctx_pa: 0x%llx\n", qpsh->hw_host_ctx_pa);
	dbg_vsnprintf("    shadow_area_pa: 0x%llx\n", qpsh->shadow_area_pa);
	dbg_vsnprintf("    q2_pa: 0x%llx\n", qpsh->q2_pa);
	dbg_vsnprintf("    dev: 0x%p\n", qpsh->dev);
	dbg_vsnprintf("    pd: 0x%p\n", qpsh->pd);
	dbg_vsnprintf("    hw_host_ctx: 0x%p\n", qpsh->hw_host_ctx);
	dbg_vsnprintf("    llp_stream_handle: 0x%p\n", qpsh->llp_stream_handle);
	dbg_vsnprintf("    back_qp: 0x%p\n", qpsh->back_qp);
	/*dbg_vsnprintf("    srq: 0x%p\n", qpsh->srq); */
	dbg_vsnprintf("    q2_buf: 0x%p\n", qpsh->q2_buf);
	dbg_vsnprintf("    qp_compl_ctx: 0x%llx\n", qpsh->qp_compl_ctx);
	dbg_vsnprintf("    qs_handle: 0x%x\n", qpsh->qs_handle);
	dbg_vsnprintf("    exception_lan_queue: 0x%x\n",
		      qpsh->exception_lan_queue);
	dbg_vsnprintf("    sq_tph_val: 0x%x\n", qpsh->sq_tph_val);
	dbg_vsnprintf("    rq_tph_val: 0x%x\n", qpsh->rq_tph_val);
	dbg_vsnprintf("    sq_tph_en: 0x%x\n", qpsh->sq_tph_en);
	dbg_vsnprintf("    rq_tph_en: 0x%x\n", qpsh->rq_tph_en);
	dbg_vsnprintf("    rcv_tph_en: 0x%x\n", qpsh->rcv_tph_en);
	dbg_vsnprintf("    xmit_tph_en: 0x%x\n", qpsh->xmit_tph_en);

	dbg_vsnprintf("    virtual_map: 0x%x\n", qpsh->virtual_map);
	dbg_vsnprintf("    flush_sq: 0x%x\n", qpsh->flush_sq);
	dbg_vsnprintf("    flush_rq: 0x%x\n", qpsh->flush_rq);
	dbg_vsnprintf("    qp_type: 0x%x\n", qpsh->qp_type);
	dbg_vsnprintf("    hw_sq_size: 0x%x\n", qpsh->hw_sq_size);
	dbg_vsnprintf("    hw_rq_size: 0x%x\n", qpsh->hw_rq_size);
}

/**
 * dump_qp_info -
 * @qp:
 **/
static void dump_qp_info(struct i40iw_qp *qp)
{
	dbg_vsnprintf("%p (struct i40iw_qp)\n", qp);
	dbg_vsnprintf("  ib_qp: 0x%p\n", &qp->ibqp);
	dbg_vsnprintf("  sc_qp: 0x%p (struct i40iw_sc_qp)\n", &qp->sc_qp);

	dump_qp_shared(qp);

	dbg_vsnprintf("  iwdev: 0x%p\n", qp->iwdev);
	dbg_vsnprintf("  iwscq: 0x%p\n", qp->iwscq);
	dbg_vsnprintf("  iwrcq: 0x%p\n", qp->iwrcq);
	dbg_vsnprintf("  iwpd: 0x%p\n", qp->iwpd);
	dbg_vsnprintf("  ctx_info: 0x%p\n", &qp->ctx_info);
	dbg_vsnprintf("  iwarp_info: 0x%p\n", &qp->iwarp_info);
	dbg_vsnprintf("  allocated_buffer: 0x%p\n", qp->allocated_buffer);
	dbg_vsnprintf("  refcount: 0x%x\n", qp->refcount.counter);
	dbg_vsnprintf("  cm_id: 0x%p\n", qp->cm_id);
	dbg_vsnprintf("  cm_node: 0x%p\n", qp->cm_node);
	dbg_vsnprintf("  lsmm_mr: 0x%p\n", qp->lsmm_mr);
	dbg_vsnprintf("  work: 0x%p\n", &qp->work);
	dbg_vsnprintf("  ibqp_state: 0x%x\n", qp->ibqp_state);
	dbg_vsnprintf("  iwarp_state: 0x%x\n", qp->iwarp_state);
	dbg_vsnprintf("  qp_mem_size: 0x%x\n", qp->qp_mem_size);
	dbg_vsnprintf("  last_aeq: 0x%x\n", qp->last_aeq);
	dbg_vsnprintf("  close_timer_started: 0x%x\n",
		      qp->close_timer_started.counter);
	dbg_vsnprintf("  qp_mem_size: 0x%x\n", qp->qp_mem_size);
	dbg_vsnprintf("  lock: 0x%p\n", &qp->lock);
	dbg_vsnprintf("  iwqp_context: 0x%p\n", qp->iwqp_context);
	dbg_vsnprintf("  pbl_vbase: 0x%p\n", qp->pbl_vbase);
	dbg_vsnprintf("  pbl_pbase: 0x%llx\n", qp->pbl_pbase);
	dbg_vsnprintf("  page: 0x%p\n", qp->page);
	dbg_vsnprintf("  terminate_timer: 0x%p\n", &qp->terminate_timer);
	dbg_vsnprintf("  terminate_eventtype: 0x%llx\n", qp->pbl_pbase);
	dbg_vsnprintf("  active_conn: %d\n", qp->active_conn);
	dbg_vsnprintf("  user_mode: %d\n", qp->user_mode);
	dbg_vsnprintf("  hte_added: %d\n", qp->hte_added);
	dbg_vsnprintf("  flush_issued: %d\n", qp->flush_issued);
	dbg_vsnprintf("  destroyed: %d\n", qp->destroyed);
	dbg_vsnprintf("  sig_all: %d\n", qp->sig_all);
	dbg_vsnprintf("  pau_mode: %d\n", qp->pau_mode);
	dbg_vsnprintf("  term_sq_flush_code: 0x%x\n", qp->term_sq_flush_code);
	dbg_vsnprintf("  term_rq_flush_code: 0x%x\n", qp->term_rq_flush_code);
	dbg_vsnprintf("  hw_iwarp_state: %d\n", qp->hw_iwarp_state);
	dbg_vsnprintf("  hw_tcp_state: %d\n", qp->hw_tcp_state);
	dbg_vsnprintf("  host_ctx.va: 0x%p pa: 0x%llx size: 0x%x\n",
		      qp->host_ctx.va, qp->host_ctx.pa, qp->host_ctx.size);
	dbg_vsnprintf("  iwpbl: 0x%p\n", qp->iwpbl);
	dbg_vsnprintf("  q2_ctx_mem.va: 0x%p pa: 0x%llx size: 0x%x\n",
		      qp->q2_ctx_mem.va, qp->q2_ctx_mem.pa,
		      qp->q2_ctx_mem.size);
	dbg_vsnprintf("  ietf_mem.va: 0x%p pa: 0x%llx size: 0x%x\n",
		      qp->ietf_mem.va, qp->ietf_mem.pa, qp->ietf_mem.size);
}

/**
 * i40iw_printqp_context -
 * @iwdev:
 * @iwqp:
 * @freeze:
 * @raw:
 **/
static void i40iw_printqp_context(struct i40iw_device *iwdev,
				  struct i40iw_qp *iwqp,
				  bool freeze,
				  bool raw)
{
	struct i40iw_dma_mem dma_mem;
	int ret = I40IW_SUCCESS;
	struct i40iw_sc_dev *dev;
	struct i40iw_sc_qp *qp;
	struct i40iw_cqp *iwcqp;
	struct i40iw_cqp_request *cqp_request;
	struct cqp_commands_info *cqp_info;
	struct i40iw_upload_context_info *info;
	u32 *ctx;

	if (!iwdev)
		return;
	qp = &iwqp->sc_qp;
	dev = &iwdev->sc_dev;

	iwcqp = &iwdev->cqp;

	cqp_request = i40iw_get_cqp_request(iwcqp, true);
	if (!cqp_request)
		return;

	cqp_info = &cqp_request->info;
	info = &cqp_info->in.u.qp_upload_context.info;
	memset(info, 0, sizeof(struct i40iw_upload_context_info));

	cqp_info->cqp_cmd = OP_QP_UPLOAD_CONTEXT;
	cqp_info->post_sq = 1;
	cqp_info->in.u.qp_upload_context.dev = dev;
	cqp_info->in.u.qp_upload_context.scratch = (u64)cqp_request;

	ret = i40iw_allocate_dma_mem(dev->hw, &dma_mem, 4096, PAGE_SIZE);
	if (ret != I40IW_SUCCESS) {
		i40iw_free_cqp_request(&iwdev->cqp, cqp_request);
		i40iw_pr_err(" mem alloc\n");
		return;
	}

	ctx = (u32 *)dma_mem.va;
	info->buf_pa = dma_mem.pa;
	info->raw_format = raw;
	info->freeze_qp = freeze;
	info->qp_type = qp->qp_type;	/* 1 is iWARP and 2 UDA */
	info->qp_id = qp->qp_uk.qp_id;

	ret = i40iw_handle_cqp_op(iwdev, cqp_request);
	if (ret != I40IW_SUCCESS) {
		i40iw_pr_err(" upload_context command failed\n");
		goto error;
	}

	i40iw_pr_info("PRINT CONTXT\n");
	{
		u32 i, j;

		for (i = 0, j = 0; i < 32; i++, j += 4)
			pr_info("%d:\t [%08X %08x %08X %08X]\n",
				(j * 4), ctx[j], ctx[j + 1], ctx[j + 2],
				ctx[j + 3]);
	}
 error:
	i40iw_free_dma_mem(dev->hw, &dma_mem);
}

/**
 * dump_qp -
 * @hdl:
 * @qp_id:
 * @subtype:
 **/
static void dump_qp(struct i40iw_handler *hdl, u32 qp_id, u32 subtype)
{
	struct i40iw_device *iwdev;
	struct i40iw_qp *iwqp = NULL;
	size_t lendumped = 0;

	iwdev = &hdl->device;

	if ((qp_id >= 2) && (qp_id < iwdev->max_qp))
		iwqp = hdl->device.qp_table[qp_id];

	if (!iwqp) {
		dbg_vsnprintf("QP %d is not valid\n", qp_id);
		return;
	}

	dbg_vsnprintf("QP %d ibqp_state=0x%x warp_state=0x%x\n", qp_id,
		      iwqp->ibqp_state, iwqp->iwarp_state);

	switch (subtype) {
	case QP_INFO:
		dump_qp_info(iwqp);
		break;
	case QP_Q2:
		dbg_vsnprintf("Q2 area 0x%p\n", iwqp->sc_qp.q2_buf);

		dump_to_buffer((char *)iwqp->sc_qp.q2_buf, 256, 0, 16, 1,
				     true, &lendumped);
		break;
	case QP_CHIP_CTX:
		dbg_vsnprintf
		    ("QP %d context will be dumped to /var/log/messages\n",
		     qp_id);
		i40iw_printqp_context(iwdev, iwqp, 0, 0);
		break;
	case QP_CHIP_RAW:
		dbg_vsnprintf
		    ("QP %d context will be dumped to /var/log/messages\n",
		     qp_id);
		i40iw_printqp_context(iwdev, iwqp, 0, 1);
		break;
	case QP_ACTIVE:
		break;
	}
}

/**
 * dump_qp_cmd -
 * @hdl:
 * @cbuf:
 **/
static void dump_qp_cmd(struct i40iw_handler *hdl, char *cbuf)
{
	static int qp_id;
	static u32 all;
	static u32 subtype = QP_ACTIVE;
	char stype[9];
	char qpstr[11];
	int offset;
	int rc;

	if (cmdnew) {
		if (sscanf(cbuf, "%10s%n", qpstr, &offset) == 0)
			return;

		if (strncasecmp(qpstr, "all", 3) == 0) {
			all = true;
			qp_id = find_next_qp_num(&hdl->device, -1);
		} else {
			all = false;
			rc = kstrtoul(qpstr, 10, (long *)&qp_id);
			if (rc)
				return;

			if ((qp_id == 0) || (qp_id == 1))
				qp_id = -1;
		}

		if (sscanf(cbuf + offset, "%8s", stype) == 1) {
			if (strncasecmp(stype, "active", 6) == 0)
				subtype = QP_ACTIVE;
			if (strncasecmp(stype, "info", 5) == 0)
				subtype = QP_INFO;
			else if (strncasecmp(stype, "q2", 2) == 0)
				subtype = QP_Q2;
			else if (strncasecmp(stype, "chip_ctx", 8) == 0)
				subtype = QP_CHIP_CTX;
			else if (strncasecmp(stype, "chip_raw", 8) == 0)
				subtype = QP_CHIP_RAW;
		}
	} else {
		if (!all)
			return;

		qp_id = find_next_qp_num(&hdl->device, qp_id);
	}

	if (qp_id >= 0)
		dump_qp(hdl, qp_id, subtype);
	else
		cmddone = true;
}

/**
 * dump_cqp_cmd -
 * @hdl:
 * @cbuf:
 **/
static void dump_cqp_cmd(struct i40iw_handler *hdl, char *cbuf)
{
	struct i40iw_sc_cqp *cqp = hdl->device.sc_dev.cqp;
	struct i40iw_device *iwdev = &hdl->device;

	if (!cqp) {
		dev_warn(iwdev->iwibdev->ibdev.dma_device,
			 "%s: cqp isn't available: cqp = NULL\n", __func__);
		return;
	}

	dbg_vsnprintf("CQP (i40iw_cqp)\n");
	dbg_vsnprintf("  size: 0x%x\n", cqp->size);
	dbg_vsnprintf("  sq_pa: 0x%llx\n", cqp->sq_pa);
	dbg_vsnprintf("  host_ctx_pa: 0x%llx\n", cqp->host_ctx_pa);
	dbg_vsnprintf("  back_cqp: 0x%p\n", cqp->back_cqp);
	dbg_vsnprintf("  dev: 0x%p\n", cqp->dev);
	dbg_vsnprintf("  sq_base: 0x%p\n", cqp->sq_base);
	dbg_vsnprintf("  host_ctx: 0x%p\n", cqp->host_ctx);
	dbg_vsnprintf("  scratch_array: 0x%p\n", cqp->scratch_array);
	dbg_vsnprintf("  cqp_id: 0x%x\n", cqp->cqp_id);
	dbg_vsnprintf("  sq_size: 0x%x\n", cqp->sq_size);
	dbg_vsnprintf("  hw_sq_size: 0x%x\n", cqp->hw_sq_size);
	dbg_vsnprintf("  struct_ver: 0x%x\n", cqp->struct_ver);
	dbg_vsnprintf("  polarity: 0x%x\n", cqp->polarity);
	dbg_vsnprintf("  en_datacenter_tcp: 0x%x\n", cqp->en_datacenter_tcp);
	dbg_vsnprintf("  sq_ring.head: 0x%x tail: 0x%x size: 0x%x\n",
		      cqp->sq_ring.head, cqp->sq_ring.tail, cqp->sq_ring.size);
	cmddone = true;
}

/**
 * dump_hmc_bp -
 * @bp:
 * @prefix:
 **/
static void dump_hmc_bp(struct i40iw_hmc_bp *bp, char *prefix)
{
	dbg_vsnprintf("%sentry_type: %d\n", prefix, bp->entry_type);

	if (bp->entry_type <= I40IW_SD_TYPE_DIRECT) {
		i40iw_dbg_dump_data_len--;
		dbg_vsnprintf(" %s\n", sd_type[bp->entry_type]);
	}

	dma_mem(&bp->addr, "addr", "    ");
	dbg_vsnprintf("%spd_index: 0x%x\n", prefix, bp->sd_pd_index);
	dbg_vsnprintf("%sref_cnt: %d\n", prefix, bp->ref_cnt);
}

/**
 * dump_hmc_pd_entry -
 * @pd_entry:
 **/
static void dump_hmc_pd_entry(struct i40iw_hmc_pd_entry *pd_entry)
{
	int i;

	dbg_vsnprintf("\npd_table 0x%p\n", pd_entry);

	for (i = 0; i < 512; i++, pd_entry++) {
		if (!pd_entry->valid)
			continue;

		dump_hmc_bp(&pd_entry->bp, "  ");

		dbg_vsnprintf("%-3d\n", i);
		dbg_vsnprintf("  sd_index=%d\n", pd_entry->sd_index);
		dbg_vsnprintf("  valid: %d\n", pd_entry->valid);
		dbg_vsnprintf("  vmem: %d\n", pd_entry->rsrc_pg);
	}
}

/**
 * dump_hmc_pd_table -
 * @pd_table:
 * @prefix:
 **/
static void dump_hmc_pd_table(struct i40iw_hmc_pd_table *pd_table, char *prefix)
{
	dma_mem(&pd_table->pd_page_addr, "pd_page_addr", "    ");
	dbg_vsnprintf("%spd_entry: 0x%p - see below\n", prefix,
		      pd_table->pd_entry);
	virt_mem(&pd_table->pd_entry_virt_mem, "pd_entry_virt_mem", "  ");
	dump_hmc_pd_entry(pd_table->pd_entry);
}

/**
 * dump_hmc_sd -:
 * @info:
**/
static bool dump_hmc_sd(struct i40iw_hmc_info *info)
{
	struct i40iw_hmc_sd_entry *sd_entry;
	static int sd_indx;
	static bool hmc_sd_new = true;

	if (hmc_sd_new) {
		sd_indx = 0;

		dbg_vsnprintf("\nsd_table\n");

		virt_mem(&info->sd_table.addr, "addr", "  ");
		dbg_vsnprintf("  sd_cnt=0x%x\n", info->sd_table.sd_cnt);
		dbg_vsnprintf("  ref_cnt=%d\n", info->sd_table.ref_cnt);
		hmc_sd_new = false;
	}

	while ((sd_indx < info->sd_table.sd_cnt) &&
	       (i40iw_dbg_dump_data_len < I40IW_DUMP_BUF_HALF_FULL)) {
		sd_entry = &info->sd_table.sd_entry[sd_indx];
		if (sd_entry && sd_entry->valid) {
			dbg_vsnprintf("\nsd_entry 0x%x (0x%x)\n", sd_indx,
				      (sd_indx * 2 * 1024 * 1024));
			dbg_vsnprintf("  entry_type: %d\n",
				      sd_entry->entry_type);

			if (sd_entry->entry_type <= I40IW_SD_TYPE_DIRECT) {
				i40iw_dbg_dump_data_len--;
				dbg_vsnprintf(" %s\n",
					      sd_type[sd_entry->entry_type]);
			}

			dbg_vsnprintf("  valid: %d\n", sd_entry->valid);

			switch (sd_entry->entry_type) {
			case I40IW_SD_TYPE_INVALID:
				break;
			case I40IW_SD_TYPE_PAGED:
				dbg_vsnprintf("  u.pd_table\n");
				dump_hmc_pd_table(&sd_entry->u.pd_table,
						  "    ");
				break;
			case I40IW_SD_TYPE_DIRECT:
				dbg_vsnprintf("  u.bp\n");
				dump_hmc_bp(&sd_entry->u.bp, "    ");
				break;
			}
		}

		sd_indx++;
	}

	if (sd_indx >= info->sd_table.sd_cnt) {
		hmc_sd_new = true;
		return true;
	}

	return false;
}

/**
 * dump_hmc_info -:
 * @info:
**/
static bool dump_hmc_info(struct i40iw_hmc_info *info)
{
	struct i40iw_hmc_obj_info *obj_info;
	size_t lendumped;
	int i;
	static char *addr;
	static size_t len;
	static int offset;
	static int hmc_flags;

	if (cmdnew)
		hmc_flags = DUMP_HMC_ALL;

	if (hmc_flags & DUMP_HMC_INFO) {
		dbg_vsnprintf("hmc_info\n");
		dbg_vsnprintf("  hmc_fn_id: 0x%x\n", info->hmc_fn_id);
		dbg_vsnprintf("  first_sd_index: 0x%x\n", info->first_sd_index);
		dbg_vsnprintf("  hmc_obj: 0x%p\n", info->hmc_obj);
		virt_mem(&info->hmc_obj_virt_mem, "hmc_obj_virt_mem", "  ");
		dbg_vsnprintf("  sd_table - see below\n");
		dbg_vsnprintf("\n  sd_indexes\n");

		addr = (char *)info->sd_indexes;
		len = I40IW_HMC_MAX_SD_COUNT * sizeof(u16);
		offset = 0;
		hmc_flags &= ~DUMP_HMC_INFO;
	}

	if (hmc_flags & DUMP_HMC_INDEXES) {
		dump_to_buffer(addr, len, offset, 16, 2, 0, &lendumped);

		addr += lendumped;
		offset = lendumped;
		len -= lendumped;
		if (len == 0)
			hmc_flags &= ~DUMP_HMC_INDEXES;
		return false;
	}

	if (hmc_flags & DUMP_HMC_OBJ_INFO) {
		obj_info = info->hmc_obj;
		if (obj_info) {
			dbg_vsnprintf("\nhmc_obj_info\n");
			for (i = 0; i < I40IW_HMC_IW_MAX; i++) {
				dbg_vsnprintf("\n  obj_info %d %s\n", i,
					      obj_info_type[i]);
				dbg_vsnprintf("    base: 0x%llx\n",
					      obj_info->base);
				dbg_vsnprintf("    max_cnt: 0x%x\n",
					      obj_info->max_cnt);
				dbg_vsnprintf("    cnt: 0x%x\n", obj_info->cnt);
				dbg_vsnprintf("    size: 0x%llx\n",
					      obj_info->size);
				obj_info++;
			}
		}

		hmc_flags &= ~DUMP_HMC_OBJ_INFO;
	}

	if (hmc_flags & DUMP_HMC_SD) {
		cmddone = dump_hmc_sd(info);
		if (cmddone)
			hmc_flags &= ~DUMP_HMC_SD;
	}

	if (hmc_flags == 0) {
		hmc_flags = DUMP_HMC_ALL;
		return true;
	}

	return false;
}

/**
 * dump_all_hmcs -
 * @hdl:
 **/
static void dump_all_hmcs(struct i40iw_handler *hdl)
{
	struct i40iw_hmc_info *info;
	static u32 vf_hmc_idx;
	static bool pf_done;

	if (cmdnew) {
		vf_hmc_idx = 0;
		dbg_vsnprintf("hmc info PF\n");
		pf_done = false;
	}

	if (!pf_done) {
		pf_done = dump_hmc_info(hdl->device.sc_dev.hmc_info);
		return;
	}

	while (vf_hmc_idx < I40IW_MAX_PE_ENABLED_VF_COUNT) {
		if (hdl->device.sc_dev.vf_dev[vf_hmc_idx]) {
			info = &hdl->device.sc_dev.vf_dev[vf_hmc_idx]->hmc_info;
			dbg_vsnprintf("hmc info VF vf_idx 0x%x\n", vf_hmc_idx);
			if (dump_hmc_info(info))
				vf_hmc_idx++;
			return;
		}

		vf_hmc_idx++;
	}
}

/**
 * dump_hmc_cmd -
 * @hdl:
 * @cbuf:
 **/
static void dump_hmc_cmd(struct i40iw_handler *hdl, char *cbuf)
{

	if (!hdl->device.sc_dev.hmc_info) {
		dbg_vsnprintf("hmc_info is not available\n");
		return;
	}

	if (strncasecmp(cbuf, "info", 4) == 0) {
		dump_all_hmcs(hdl);
	} else {
		dump_help(hdl);
	}
}

/**
 * dump_pble_info -:
 * @leaf:
 * @fbo:
 * @type:
 * @prefix:
**/
static bool dump_pble_info(struct i40iw_pble_info *leaf,
			   u32 fbo,
			   char *type,
			   char *prefix)
{
	u64 *paddrs = (u64 *)leaf->addr;
	static int pbleidx;
	static u64 data_offset;

	if (pbleidx == 0)
		dbg_vsnprintf("\n%s%s has 0x%x pbles\n", prefix, type,
			      leaf->cnt);

	if (cmdnew) {
		/* Prepare to dump list of data pages */
		dbg_vsnprintf
		    ("%s  pble-indx  pble-phys-addr       data-phys-addr      data-offset         hmc-idx\n",
		     prefix);

		/* Print first one here to add in the fbo */
		dbg_vsnprintf
		    ("%s  0x%-8x 0x%-16lx   0x%-16lx  0x%-16lx  0x%x\n", prefix,
		     0, virt_to_bus(paddrs), paddrs[0] + fbo, 0l,
		     leaf->idx);

		data_offset = 4096 - fbo;
		pbleidx = 1;
	}

	while (pbleidx < leaf->cnt) {
		dbg_vsnprintf
		    ("%s  0x%-8x 0x%-16lx   0x%-16lx  0x%-16lx  0x%x\n", prefix,
		     pbleidx, virt_to_bus(&paddrs[pbleidx]), paddrs[pbleidx],
		     data_offset, leaf->idx + pbleidx);
		data_offset += 4096;
		pbleidx++;

		if (i40iw_dbg_dump_data_len > I40IW_DUMP_BUF_HALF_FULL)
			return false;
	}

	pbleidx = 0;

	return true;
}

/**
 * dump_pble_root -:
 * @lvl2:
 * @fbo:
 * @prefix:
**/
static bool dump_pbl_root(struct i40iw_pble_level2 *lvl2, u32 fbo, char *prefix)
{
	static int leafidx;
	static struct i40iw_pble_info *leaf;
	char info[20];

	if (cmdnew) {
		leaf = lvl2->leaf;
		leafidx = 0;

		dbg_vsnprintf
		    ("%s root hmc-idx=0x%x paddr 0x%lx has 0x%x leaves\n",
		     prefix, lvl2->root.idx, virt_to_bus(&lvl2->root.addr),
		     lvl2->leaf_cnt);
	}

	while (leafidx < lvl2->leaf_cnt) {
		snprintf(info, sizeof(info), "leaf 0x%x", leafidx);
		if (dump_pble_info(leaf, fbo, info, "   ") == false)
			return false;

		leaf++;
		leafidx++;
	}

	return true;
}

/**
 * dump_mr -:
 * @ibmr:
 * @kernel_mr:
 * @vf_idx:
 * @show_pbles:
**/
static bool dump_mr(struct ib_mr *ibmr, bool kernel_mr, int vf_idx, bool show_pbles)
{
	u32 fbo;
	struct i40iw_mr *iwmr = to_iwmr(ibmr);
	int lvl;
	bool done = true;

	if (!iwmr->iwpbl.pbl_allocated)
		lvl = 0;
	else if (iwmr->iwpbl.pble_alloc.level == I40IW_LEVEL_1)
		lvl = 1;
	else
		lvl = 2;

	if (cmdnew || !show_pbles) {
		dbg_vsnprintf
		    ("stag 0x%-8x len 0x%-8x page_cnt 0x%-8x type %-4s level%d user_base 0x%-16lx ",
		     iwmr->stag, iwmr->length, iwmr->page_cnt,
		     mr_types[iwmr->type], lvl, iwmr->iwpbl.user_base);

		if (kernel_mr)
			dbg_vsnprintf("kernel ");
		else
			dbg_vsnprintf("user   ");

		if (vf_idx == -1)
			dbg_vsnprintf("PF-hmc\n");
		else
			dbg_vsnprintf("VF-hmc index %d\n", vf_idx);
	}

	if (show_pbles) {
		if (cmdnew)
			fbo = (u32)(iwmr->iwpbl.user_base & OFFSET_MASK_4K);
		else
			fbo = 0;

		if (lvl == 0)
			dbg_vsnprintf("  level0 pble pa 0x%016lx\n",
				      iwmr->level0_pa + fbo);
		else if (lvl == 1)
			done = dump_pble_info(&iwmr->iwpbl.pble_alloc.level1,
					      fbo, "level1", "  ");
		else
			done = dump_pbl_root(&iwmr->iwpbl.pble_alloc.level2,
					     fbo, "  ");
	}

	return done;
}

/**
 * dump_mr_cmds -
 * @hdl:
 * @cmd:
 * @paddr:
 * @stag:
 * @vf_idx:
 **/
static void dump_mr_cmds(struct i40iw_handler *hdl, enum mr_cmd_type cmd,
			 dma_addr_t paddr, u32 stag, int vf_idx)
{
	static struct ib_uobject *uobj;
	static struct i40iw_ucontext *ucontext;
	struct ib_mr *ibmr = uobj->object;

	if (cmdnew) {
		if (list_empty(&hdl->ucontext_list))
			goto dump_mr_done;

		ucontext = list_first_entry(&hdl->ucontext_list,
					    struct i40iw_ucontext, uctx_list);

		uobj = list_first_entry(&ucontext->ibucontext.mr_list,
					struct ib_uobject, list);
	}

	while (1) {
		while (1) {
			if (i40iw_dbg_dump_data_len > I40IW_DUMP_BUF_HALF_FULL)
				return;

			if (list_empty(&ucontext->ibucontext.mr_list))
				break;

			ibmr = uobj->object;

			switch (cmd) {
			case MR_CMD_LIST:
				dump_mr(ibmr, false, vf_idx, false);
				break;
			case MR_CMD_STAG:
				if (stag == ibmr->lkey) {
					if (!dump_mr(ibmr, false, vf_idx, true))
						return;
					goto dump_mr_done;
				}
			}

			if (list_is_last
			    (&uobj->list, &ucontext->ibucontext.mr_list))
				break;
			uobj =
			    list_entry(uobj->list.next, struct ib_uobject,
				       list);
		}

		if (list_is_last(&ucontext->uctx_list, &hdl->ucontext_list))
			break;
		ucontext =
		    list_entry(ucontext->uctx_list.next, struct i40iw_ucontext,
			       uctx_list);
		uobj =
		    list_first_entry(&ucontext->ibucontext.mr_list,
				     struct ib_uobject, list);
	}

	if (cmd != MR_CMD_LIST)
		dbg_vsnprintf("not found\n");

 dump_mr_done:
	cmddone = true;
}

/**
 * dump_mr_cmd -
 * @hdl:
 * @cbuf:
 **/
static void dump_mr_cmd(struct i40iw_handler *hdl, char *cbuf)
{
	static dma_addr_t paddr;
	static u32 stag;
	static enum mr_cmd_type cmd;
	int rc;

	if (cmdnew) {
		if (strncasecmp(cbuf, "list", 4) == 0) {
			cmd = MR_CMD_LIST;
		} else if (strncasecmp(cbuf, "stag ", 5) == 0) {
			rc = kstrtoul(cbuf + 5, 16, (long *)&stag);
			if (rc)
				return;
			cmd = MR_CMD_STAG;
		} else {
			dump_help(hdl);
			return;
		}
	}

	dump_mr_cmds(hdl, cmd, paddr, stag, -1);
}

/**
 * dump_stats_cmd -
 * @hdl:
 **/
static void dump_stats_cmd(struct i40iw_handler *hdl)
{
	struct i40iw_device *iwdev = &hdl->device;
	struct i40iw_sc_dev *dev = &iwdev->sc_dev;
	struct i40iw_cm_core *cm_core = &iwdev->cm_core;
	struct i40iw_hmc_pble_rsrc *pble = iwdev->pble_rsrc;
	struct i40iw_puda_rsrc *ilq = iwdev->vsi.ilq;

	dbg_vsnprintf("cm nodes created                 %d\n",
		      cm_core->stats_nodes_created);
	dbg_vsnprintf("cm nodes destroyed               %d\n",
		      cm_core->stats_nodes_destroyed);
	dbg_vsnprintf("cm listen called                 %d\n",
		      cm_core->stats_listen_created);
	dbg_vsnprintf("cm listen removed                %d\n",
		      cm_core->stats_listen_destroyed);
	dbg_vsnprintf("cm listen nodes created          %d\n",
		      cm_core->stats_listen_nodes_created);
	dbg_vsnprintf("cm listen nodes destroyed        %d\n",
		      cm_core->stats_listen_nodes_destroyed);
	dbg_vsnprintf("cm accepts                       %d\n",
		      cm_core->stats_accepts);
	dbg_vsnprintf("cm rejects                       %d\n",
		      cm_core->stats_rejects);
	dbg_vsnprintf("cm loopbacks                     %d\n",
		      cm_core->stats_loopbacks);
	dbg_vsnprintf("cm connect errors                %d\n",
		      cm_core->stats_connect_errs);
	dbg_vsnprintf("cm passive errors                %d\n",
		      cm_core->stats_passive_errs);
	dbg_vsnprintf("cm pkts retrans                  %d\n",
		      cm_core->stats_pkt_retrans);
	dbg_vsnprintf("cm backlog drops                 %d\n",
		      cm_core->stats_backlog_drops);

	dbg_vsnprintf("pble direct sds                  %d\n",
		      pble->stats_direct_sds);
	dbg_vsnprintf("pble paged sds                   %d\n",
		      pble->stats_paged_sds);
	dbg_vsnprintf("pble alloc ok                    %d\n",
		      pble->stats_alloc_ok);
	dbg_vsnprintf("pble alloc fail                  %d\n",
		      pble->stats_alloc_fail);
	dbg_vsnprintf("pble alloc freed                 %d\n",
		      pble->stats_alloc_freed);
	dbg_vsnprintf("pble lvl1 alloc                  %d\n",
		      pble->stats_lvl1);
	dbg_vsnprintf("pble lvl2 alloc                  %d\n",
		      pble->stats_lvl2);

	dbg_vsnprintf("hugepages registered             %d\n",
		      iwdev->hugepgcnt);

	dbg_vsnprintf("ilq packet sent                  %d\n",
		      ilq->stats_pkt_sent);
	dbg_vsnprintf("ilq buf alloc fail               %d\n",
		      ilq->stats_buf_alloc_fail);
	dbg_vsnprintf("ilq packet rcvd                  %d\n",
		      ilq->stats_pkt_rcvd);
	dbg_vsnprintf("ilq packet rcv error             %d\n",
		      ilq->stats_rcvd_pkt_err);

	dbg_vsnprintf("cqp OP_QP_CREATE                 %d\n",
		      dev->cqp_cmd_stats[OP_QP_CREATE]);
	dbg_vsnprintf("cqp OP_QP_MODIFY                 %d\n",
		      dev->cqp_cmd_stats[OP_QP_MODIFY]);
	dbg_vsnprintf("cqp OP_QP_DESTROY                %d\n",
		      dev->cqp_cmd_stats[OP_QP_DESTROY]);
	dbg_vsnprintf("cqp OP_CQ_CREATE                 %d\n",
		      dev->cqp_cmd_stats[OP_CQ_CREATE]);
	dbg_vsnprintf("cqp OP_CQ_DESTROY                %d\n",
		      dev->cqp_cmd_stats[OP_CQ_DESTROY]);
	dbg_vsnprintf("cqp OP_ALLOC_STAG                %d\n",
		      dev->cqp_cmd_stats[OP_ALLOC_STAG]);
	dbg_vsnprintf("cqp OP_MR_REG_NON_SHARED         %d\n",
		      dev->cqp_cmd_stats[OP_MR_REG_NON_SHARED]);
	dbg_vsnprintf("cqp OP_DEALLOC_STAG              %d\n",
		      dev->cqp_cmd_stats[OP_DEALLOC_STAG]);
	dbg_vsnprintf("cqp OP_ADD_ARP_CACHE_ENTRY       %d\n",
		      dev->cqp_cmd_stats[OP_ADD_ARP_CACHE_ENTRY]);
	dbg_vsnprintf("cqp OP_DELETE_ARP_CACHE_ENTRY    %d\n",
		      dev->cqp_cmd_stats[OP_DELETE_ARP_CACHE_ENTRY]);
	dbg_vsnprintf("cqp OP_DELETE_LOCAL_MAC_IPADDR_ENTRY %d\n",
		      dev->cqp_cmd_stats[OP_DELETE_LOCAL_MAC_IPADDR_ENTRY]);
	dbg_vsnprintf("cqp OP_CEQ_DESTROY               %d\n",
		      dev->cqp_cmd_stats[OP_CEQ_DESTROY]);
	dbg_vsnprintf("cqp OP_AEQ_DESTROY               %d\n",
		      dev->cqp_cmd_stats[OP_AEQ_DESTROY]);
	dbg_vsnprintf("cqp OP_MANAGE_APBVT_ENTRY        %d\n",
		      dev->cqp_cmd_stats[OP_MANAGE_APBVT_ENTRY]);
	dbg_vsnprintf("cqp OP_CEQ_CREATE                %d\n",
		      dev->cqp_cmd_stats[OP_CEQ_CREATE]);
	dbg_vsnprintf("cqp OP_AEQ_CREATE                %d\n",
		      dev->cqp_cmd_stats[OP_AEQ_CREATE]);
	dbg_vsnprintf("cqp OP_ALLOC_LOCAL_MAC_IPADDR_ENTRY %d\n",
		      dev->cqp_cmd_stats[OP_ALLOC_LOCAL_MAC_IPADDR_ENTRY]);
	dbg_vsnprintf("cqp OP_ADD_LOCAL_MAC_IPADDR_ENTRY %d\n",
		      dev->cqp_cmd_stats[OP_ADD_LOCAL_MAC_IPADDR_ENTRY]);
	dbg_vsnprintf("cqp OP_QP_UPLOAD_CONTEXT         %d\n",
		      dev->cqp_cmd_stats[OP_QP_UPLOAD_CONTEXT]);
	dbg_vsnprintf("cqp OP_MW_ALLOC                  %d\n",
		      dev->cqp_cmd_stats[OP_MW_ALLOC]);
	dbg_vsnprintf("cqp OP_QP_FLUSH_WQES             %d\n",
		      dev->cqp_cmd_stats[OP_QP_FLUSH_WQES]);
	dbg_vsnprintf("cqp OP_MANAGE_PUSH_PAGE          %d\n",
		      dev->cqp_cmd_stats[OP_MANAGE_PUSH_PAGE]);
	dbg_vsnprintf("cqp OP_UPDATE_PE_SDS             %d\n",
		      dev->cqp_cmd_stats[OP_UPDATE_PE_SDS]);
	dbg_vsnprintf("cqp OP_MANAGE_HMC_PM_FUNC_TABLE  %d\n",
		      dev->cqp_cmd_stats[OP_MANAGE_HMC_PM_FUNC_TABLE]);
	dbg_vsnprintf("cqp OP_SUSPEND                   %d\n",
		      dev->cqp_cmd_stats[OP_SUSPEND]);
	dbg_vsnprintf("cqp OP_RESUME                    %d\n",
		      dev->cqp_cmd_stats[OP_RESUME]);
	dbg_vsnprintf("cqp OP_MANAGE_VF_PBLE_BP         %d\n",
		      dev->cqp_cmd_stats[OP_MANAGE_VF_PBLE_BP]);
	dbg_vsnprintf("cqp OP_QUERY_FPM_VALUES          %d\n",
		      dev->cqp_cmd_stats[OP_QUERY_FPM_VALUES]);
	dbg_vsnprintf("cqp OP_MANAGE_QHASH_TABLE_ENTRY  %d\n",
		      dev->cqp_cmd_stats[OP_MANAGE_QHASH_TABLE_ENTRY]);
	cmddone = true;
}

/**
 * dump_hw_stats_cmd -
 * @hdl:
 **/
static void dump_hw_stats_cmd(struct i40iw_handler *hdl)
{
	struct i40iw_device *iwdev = &hdl->device;
	struct i40iw_sc_dev *dev = &iwdev->sc_dev;
	struct i40iw_vsi_pestat *devstat = iwdev->vsi.pestat;
	struct i40iw_dev_hw_stats *hw_stats = &devstat->hw_stats;

	if (dev->is_pf) {
		i40iw_hw_stats_read_all(devstat, &devstat->hw_stats);
	} else {
		if (i40iw_vchnl_vf_get_pe_stats(dev, &devstat->hw_stats)) {
			dbg_vsnprintf("VF HW stat read failed\n");
			return;
		}
	}

	dbg_vsnprintf("IPV4 octs recvd                      %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_IP4RXOCTS]);
	dbg_vsnprintf("IPV6 octs recvd                      %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_IP6RXOCTS]);
	dbg_vsnprintf("IPV4 pkts recvd                      %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_IP4RXPKTS]);
	dbg_vsnprintf("IPV6 pkts recvd                      %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_IP6RXPKTS]);
	dbg_vsnprintf("IPV4 frag recvd                      %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_IP4RXFRAGS]);
	dbg_vsnprintf("IPV6 frag recvd                      %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_IP6RXFRAGS]);
	dbg_vsnprintf("IPV4 multicast pkts recvd            %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_IP4RXMCPKTS]);
	dbg_vsnprintf("IPV6 multicast pkts recvd            %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_IP6RXMCPKTS]);
	dbg_vsnprintf("IPV4 octs trans                      %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_IP4TXOCTS]);
	dbg_vsnprintf("IPV6 octs trans                      %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_IP6TXOCTS]);
	dbg_vsnprintf("IPV4 pkts trans                      %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_IP4TXPKTS]);
	dbg_vsnprintf("IPV6 pkts trans                      %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_IP6TXPKTS]);
	dbg_vsnprintf("IPV4 frag trans                      %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_IP4TXFRAGS]);
	dbg_vsnprintf("IPV6 frag trans                      %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_IP6TXFRAGS]);
	dbg_vsnprintf("IPV4 multicast pkts trans            %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_IP4TXMCPKTS]);
	dbg_vsnprintf("IPV6 multicast pkts trans            %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_IP6TXMCPKTS]);
	dbg_vsnprintf("IPV4 pkts recvd & discarded          %llu\n",
		      hw_stats->stats_value_32[I40IW_HW_STAT_INDEX_IP4RXDISCARD]);
	dbg_vsnprintf("IPV6 pkts recvd & discarded          %llu\n",
		      hw_stats->stats_value_32[I40IW_HW_STAT_INDEX_IP6RXDISCARD]);
	dbg_vsnprintf("IPV4 pkts recvd & truncated          %llu\n",
		      hw_stats->stats_value_32[I40IW_HW_STAT_INDEX_IP4RXTRUNC]);
	dbg_vsnprintf("IPV6 pkts recvd & truncated          %llu\n",
		      hw_stats->stats_value_32[I40IW_HW_STAT_INDEX_IP6RXTRUNC]);
	dbg_vsnprintf("IPV4 dtgms discarded (no ARP hit)    %llu\n",
		      hw_stats->stats_value_32[I40IW_HW_STAT_INDEX_IP4TXNOROUTE]);
	dbg_vsnprintf("IPV6 dtgms discarded (no ARP hit)    %llu\n",
		      hw_stats->stats_value_32[I40IW_HW_STAT_INDEX_IP6TXNOROUTE]);

	dbg_vsnprintf("TCP sgmnts recvd                     %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_TCPRXSEGS]);
	dbg_vsnprintf("TCP sgmnts trans                     %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_TCPTXSEG]);
	dbg_vsnprintf("TCP sgmnts retrans                   %llu\n",
		      hw_stats->stats_value_32[I40IW_HW_STAT_INDEX_TCPRTXSEG]);
	dbg_vsnprintf("TCP sgmnts recvd w/ err              %llu\n",
		      hw_stats->stats_value_32[I40IW_HW_STAT_INDEX_TCPRXOPTERR]);
	dbg_vsnprintf("TCP sgmnts recvd w/ protocol-err      %llu\n",
		      hw_stats->stats_value_32[I40IW_HW_STAT_INDEX_TCPRXPROTOERR]);

	dbg_vsnprintf("RDMA read-req msgs recvd             %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_RDMARXRDS]);
	dbg_vsnprintf("RDMA send-type msgs recvd            %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_RDMARXSNDS]);
	dbg_vsnprintf("RDMA write msgs recvd                %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_RDMARXWRS]);
	dbg_vsnprintf("RDMA read-req msgs sent              %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_RDMATXRDS]);
	dbg_vsnprintf("RDMA send-type msgs sent             %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_RDMATXSNDS]);
	dbg_vsnprintf("RDMA write msgs sent                 %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_RDMATXWRS]);
	dbg_vsnprintf("RDMA verb-bind ops                   %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_RDMAVBND]);
	dbg_vsnprintf("RDMA verb-inv ops                    %llu\n",
		      hw_stats->stats_value_64[I40IW_HW_STAT_INDEX_RDMAVINV]);

	cmddone = true;
}

/**
 * i40iw_dbg_dump_read - read the dump data
 * @filp: the opened file
 * @buffer: where to write the data for the user to read
 * @count: the size of the user's buffer
 * @ppos: file position offset
 *
 * When a read happens, the file system will keep calling this routine
 * until it returns with no data.  This is because the buffers may be too
 * small to contain all the data at once.  Some of the cmds count on being
 * able to return the results in pieces.
 **/
static ssize_t i40iw_dbg_dump_read(struct file *filp,
				   char __user *buffer,
				   size_t count,
				   loff_t *ppos)
{
	struct i40iw_handler *hdl = filp->private_data;
	int bytes_not_copied;
	int len;
	struct i40iw_device *iwdev;

	if (cmddone) {
		cmdnew = true;
		cmddone = false;
	} else {
		if (cmdnew)
			dbg_vsnprintf("cmd: %s", cmd_buf);

		if (strncasecmp(cmd_buf, "qp ", 3) == 0)
			dump_qp_cmd(hdl, &cmd_buf[3]);
		else if (strncasecmp(cmd_buf, "cqp", 3) == 0)
			dump_cqp_cmd(hdl, &cmd_buf[3]);
		else if (strncasecmp(cmd_buf, "hmc", 3) == 0)
			dump_hmc_cmd(hdl, &cmd_buf[4]);
		else if (strncasecmp(cmd_buf, "mr ", 3) == 0)
			dump_mr_cmd(hdl, &cmd_buf[3]);
		else if (strncasecmp(cmd_buf, "sw-stats", 8) == 0)
			dump_stats_cmd(hdl);
		else if (strncasecmp(cmd_buf, "hw-stats", 8) == 0)
			dump_hw_stats_cmd(hdl);
		else
			dump_help(hdl);
		cmdnew = false;
	}

	len = min_t(int, count, (i40iw_dbg_dump_data_len - *ppos));
	if (len == 0) {
		*ppos = 0;
		i40iw_dbg_dump_data_len = 0;
		return 0;
	}

	bytes_not_copied =
	    copy_to_user(buffer, &i40iw_dbg_dump_buf[*ppos], len);
	if (bytes_not_copied) {
		iwdev = &hdl->device;
		dev_warn(iwdev->iwibdev->ibdev.dma_device,
			 "copy_to_user returned 0x%x\n", bytes_not_copied);
	}

	*ppos += len;
	if (*ppos >= i40iw_dbg_dump_data_len) {
		*ppos = 0;
		i40iw_dbg_dump_data_len = 0;
	}
	return len;
}

/**
 * i40iw_dbg_dump_write - trigger a datadump snapshot
 * @filp: the opened file
 * @buffer: where to find the user's data
 * @count: the length of the user's data
 * @ppos: file position offset
 **/
static ssize_t i40iw_dbg_dump_write(struct file *filp,
				    const char __user *buffer,
				    size_t count,
				    loff_t *ppos)
{
	int bytes_not_copied;

	/* don't allow partial writes */
	if (*ppos != 0)
		return 0;
	if (count >= sizeof(cmd_buf))
		return -ENOSPC;

	bytes_not_copied = copy_from_user(cmd_buf, buffer, count);
	if (bytes_not_copied < 0)
		return bytes_not_copied;
	if (bytes_not_copied > 0)
		count -= bytes_not_copied;

	cmd_buf[count] = '\0';
	cmdnew = true;
	cmddone = false;
	*ppos = 0;
	i40iw_dbg_dump_data_len = 0;
	return count;
}

/**
 * i40iw_dbg_prep_dump_buf
 * @hdl: the iWARP handler we're working with
 * @buflen: the desired buffer length
 *
 * Return positive if success, 0 if failed
 **/
static int i40iw_dbg_prep_dump_buf(struct i40iw_handler *hdl, int buflen)
{
	if (i40iw_dbg_dump_buffer_len && i40iw_dbg_dump_buffer_len < buflen) {
		kfree(i40iw_dbg_dump_buf);
		i40iw_dbg_dump_buffer_len = 0;
		i40iw_dbg_dump_buf = NULL;
	}

	if (!i40iw_dbg_dump_buf) {
		i40iw_dbg_dump_buf = kzalloc(buflen, GFP_KERNEL);
		if (!i40iw_dbg_dump_buf) {
			i40iw_dbg_dump_buffer_len = 0;
			pr_err("%s: memory alloc for snapshot failed\n",
			       __func__);
		} else {
			i40iw_dbg_dump_buffer_len = buflen;
			pr_err("%s: i40iw_dbg_dump_buffer_len = %d\n",
			       __func__, (int)i40iw_dbg_dump_buffer_len);
		}
	}

	return i40iw_dbg_dump_buffer_len;
}

static const struct file_operations i40iw_dbg_dump_fops = {
	.owner = THIS_MODULE,
	.open = simple_open,
	.read = i40iw_dbg_dump_read,
	.write = i40iw_dbg_dump_write,
};

/**
 * i40iw_dbg_pf_init - setup the debugfs directory for the pf
 * @hdl: the iWARP handler that is starting up
 **/
void i40iw_dbg_pf_init(struct i40iw_handler *hdl)
{
	const char *name = pci_name(hdl->ldev.pcidev);
	struct dentry *pfile __attribute__ ((unused));

	if (i40iw_dbg_prep_dump_buf(hdl, I40IW_DUMP_BUF_SIZE) == 0) {
		pr_err("i40iw_dbg_pf_init: unable to allocate debugfs dump buffer\n");
		return;
	}

	hdl->i40iw_dbg_dentry = debugfs_create_dir(name, i40iw_dbg_root);
	if (hdl->i40iw_dbg_dentry)
		pfile =
		    debugfs_create_file("dump", 0600, hdl->i40iw_dbg_dentry,
					hdl, &i40iw_dbg_dump_fops);
	else
		pr_err("%s: debugfs entry for %s failed\n", __func__, name);

	spin_lock_init(&hdl->uctx_list_lock);
	INIT_LIST_HEAD(&hdl->ucontext_list);
}

/**
 * i40iw_dbg_pf_exit - clear out the pf's debugfs entries
 * @hdl: the iWARP handler that is stopping
 **/
void i40iw_dbg_pf_exit(struct i40iw_handler *hdl)
{
	if (hdl) {
		pr_err("%s: removing debugfs entries\n", __func__);
		debugfs_remove_recursive(hdl->i40iw_dbg_dentry);
		hdl->i40iw_dbg_dentry = NULL;
	}
}

/**
 * i40iw_dbg_init - start up debugfs for the driver
 **/
void i40iw_dbg_init(void)
{
	i40iw_dbg_root = debugfs_create_dir(i40iw_driver_name, NULL);
	if (!i40iw_dbg_root)
		pr_err("%s: init of debugfs failed\n", __func__);
}

/**
 * i40iw_dbg_exit - clean out the driver's debugfs entries
 **/
void i40iw_dbg_exit(void)
{
	kfree(i40iw_dbg_dump_buf);
	i40iw_dbg_dump_buffer_len = 0;
	i40iw_dbg_dump_buf = NULL;
	debugfs_remove_recursive(i40iw_dbg_root);
}

#endif /* CONFIG_DEBUG_FS */
